/* ==========================================================================
   CPW Consentimento - Script do Banner de Consentimento
   ========================================================================== */

(function($) {
    'use strict';

    var CPWConsent = {

        cookieName: cpwConsentData.cookieName || 'cpw_consent_status',
        cookieDuration: parseInt(cpwConsentData.cookieDuration) || 365,
        ajaxUrl: cpwConsentData.ajaxUrl || '',
        nonce: cpwConsentData.nonce || '',

        /**
         * Inicializa o plugin de consentimento.
         */
        init: function() {
            var self = this;

            // Verifica se o consentimento já foi dado
            if (self.getCookie(self.cookieName) === null) {
                self.showBanner();
            }

            // Botão Aceitar Todos
            $(document).on('click', '#cpw-btn-accept', function(e) {
                e.preventDefault();
                self.acceptAll();
            });

            // Botão Recusar
            $(document).on('click', '#cpw-btn-decline', function(e) {
                e.preventDefault();
                self.declineAll();
            });

            // Botão Personalizar
            $(document).on('click', '#cpw-btn-customize', function(e) {
                e.preventDefault();
                self.openCustomizeModal();
            });

            // Fechar modal
            $(document).on('click', '#cpw-modal-close, #cpw-consent-overlay', function(e) {
                e.preventDefault();
                self.closeCustomizeModal();
            });

            // Salvar preferências no modal
            $(document).on('click', '#cpw-btn-save-preferences', function(e) {
                e.preventDefault();
                self.savePreferences();
            });

            // Aceitar tudo no modal
            $(document).on('click', '#cpw-btn-accept-all-modal', function(e) {
                e.preventDefault();
                self.acceptAll();
                self.closeCustomizeModal();
            });

            // Toggle de categorias
            $(document).on('click', '.cpw-cookie-category-header', function() {
                $(this).next('.cpw-cookie-category-body').slideToggle(200);
            });

            // Link de configurações de cookies no rodapé
            $(document).on('click', '.cpw-open-consent-settings', function(e) {
                e.preventDefault();
                self.openCustomizeModal();
            });
        },

        /**
         * Exibe o banner de consentimento.
         */
        showBanner: function() {
            $('#cpw-consent-banner').fadeIn(400);
            if ($('#cpw-consent-banner').hasClass('cpw-position-center')) {
                $('#cpw-consent-overlay').fadeIn(300);
            }
        },

        /**
         * Oculta o banner de consentimento.
         */
        hideBanner: function() {
            $('#cpw-consent-banner').fadeOut(300);
            $('#cpw-consent-overlay').fadeOut(300);
        },

        /**
         * Aceita todos os cookies.
         */
        acceptAll: function() {
            var consentData = {
                analytics_storage: 'granted',
                ad_storage: 'granted',
                ad_user_data: 'granted',
                ad_personalization: 'granted',
                functionality_storage: 'granted',
                security_storage: 'granted',
                personalization_storage: 'granted'
            };

            this.setCookie(this.cookieName, 'granted', this.cookieDuration);
            this.setCookie('cpw_consent_data', JSON.stringify(consentData), this.cookieDuration);
            this.updateGoogleConsent(consentData);
            this.logConsent('granted', consentData);
            this.hideBanner();
        },

        /**
         * Recusa todos os cookies opcionais.
         */
        declineAll: function() {
            var consentData = {
                analytics_storage: 'denied',
                ad_storage: 'denied',
                ad_user_data: 'denied',
                ad_personalization: 'denied',
                functionality_storage: 'granted',
                security_storage: 'granted',
                personalization_storage: 'denied'
            };

            this.setCookie(this.cookieName, 'denied', this.cookieDuration);
            this.setCookie('cpw_consent_data', JSON.stringify(consentData), this.cookieDuration);
            this.updateGoogleConsent(consentData);
            this.logConsent('denied', consentData);
            this.hideBanner();
        },

        /**
         * Salva as preferências personalizadas.
         */
        savePreferences: function() {
            var analyticsChecked     = $('#cpw-toggle-analytics').is(':checked');
            var advertisingChecked   = $('#cpw-toggle-advertising').is(':checked');
            var personalizationChecked = $('#cpw-toggle-personalization').is(':checked');

            var consentData = {
                analytics_storage: analyticsChecked ? 'granted' : 'denied',
                ad_storage: advertisingChecked ? 'granted' : 'denied',
                ad_user_data: advertisingChecked ? 'granted' : 'denied',
                ad_personalization: personalizationChecked ? 'granted' : 'denied',
                functionality_storage: 'granted',
                security_storage: 'granted',
                personalization_storage: personalizationChecked ? 'granted' : 'denied'
            };

            var overallStatus = (analyticsChecked || advertisingChecked || personalizationChecked) ? 'partial' : 'denied';

            this.setCookie(this.cookieName, overallStatus, this.cookieDuration);
            this.setCookie('cpw_consent_data', JSON.stringify(consentData), this.cookieDuration);
            this.updateGoogleConsent(consentData);
            this.logConsent(overallStatus, consentData);
            this.closeCustomizeModal();
            this.hideBanner();
        },

        /**
         * Abre o modal de personalização.
         */
        openCustomizeModal: function() {
            $('#cpw-customize-modal').fadeIn(300);
            $('#cpw-consent-overlay').fadeIn(300);
            $('body').css('overflow', 'hidden');

            // Pré-preencher com preferências salvas
            var savedData = this.getCookie('cpw_consent_data');
            if (savedData) {
                try {
                    var data = JSON.parse(savedData);
                    $('#cpw-toggle-analytics').prop('checked', data.analytics_storage === 'granted');
                    $('#cpw-toggle-advertising').prop('checked', data.ad_storage === 'granted');
                    $('#cpw-toggle-personalization').prop('checked', data.personalization_storage === 'granted');
                } catch(e) {}
            }
        },

        /**
         * Fecha o modal de personalização.
         */
        closeCustomizeModal: function() {
            $('#cpw-customize-modal').fadeOut(300);
            if (!$('#cpw-consent-banner').is(':visible')) {
                $('#cpw-consent-overlay').fadeOut(300);
            }
            $('body').css('overflow', '');
        },

        /**
         * Atualiza o Google Consent Mode v2.
         */
        updateGoogleConsent: function(consentData) {
            if (typeof gtag === 'function') {
                gtag('consent', 'update', consentData);
            }
            // Dispara evento customizado para integração com outros plugins
            $(document).trigger('cpw_consent_updated', [consentData]);
        },

        /**
         * Registra o consentimento via AJAX.
         */
        logConsent: function(status, consentData) {
            if (!this.ajaxUrl || !this.nonce) return;

            $.post(this.ajaxUrl, {
                action: 'cpw_log_consent',
                nonce: this.nonce,
                consent_status: status,
                consent_data: JSON.stringify(consentData)
            });
        },

        /**
         * Define um cookie.
         */
        setCookie: function(name, value, days) {
            var expires = '';
            if (days) {
                var date = new Date();
                date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
                expires = '; expires=' + date.toUTCString();
            }
            document.cookie = name + '=' + encodeURIComponent(value || '') + expires + '; path=/; SameSite=Lax';
        },

        /**
         * Obtém o valor de um cookie.
         */
        getCookie: function(name) {
            var nameEQ = name + '=';
            var ca = document.cookie.split(';');
            for (var i = 0; i < ca.length; i++) {
                var c = ca[i];
                while (c.charAt(0) === ' ') c = c.substring(1, c.length);
                if (c.indexOf(nameEQ) === 0) {
                    return decodeURIComponent(c.substring(nameEQ.length, c.length));
                }
            }
            return null;
        }
    };

    // Inicializa quando o DOM estiver pronto
    $(document).ready(function() {
        CPWConsent.init();
    });

    // Expõe a API publicamente
    window.CPWConsent = CPWConsent;

})(jQuery);
