<?php
/**
 * Classe responsável pela parte pública do plugin CPW Consentimento.
 *
 * @package CPW_Consentimento
 * @subpackage CPW_Consentimento/public
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class CPW_Public {

    private $plugin_name;
    private $version;

    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version     = $version;
    }

    // -------------------------------------------------------------------------
    // Enqueue de estilos e scripts
    // -------------------------------------------------------------------------

    public function enqueue_styles() {
        $general_options = get_option( 'cpw_general_options', array() );
        if ( empty( $general_options['enable_plugin'] ) ) {
            return;
        }
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'css/consent-banner.css',
            array(),
            $this->version,
            'all'
        );
    }

    public function enqueue_scripts() {
        $general_options = get_option( 'cpw_general_options', array() );
        if ( empty( $general_options['enable_plugin'] ) ) {
            return;
        }
        wp_enqueue_script(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'js/consent-banner.js',
            array( 'jquery' ),
            $this->version,
            true
        );

        // Passa dados para o JavaScript
        $cookie_name     = ! empty( $general_options['cookie_name'] ) ? $general_options['cookie_name'] : 'cpw_consent_status';
        $cookie_duration = ! empty( $general_options['cookie_duration'] ) ? $general_options['cookie_duration'] : 365;

        wp_localize_script( $this->plugin_name, 'cpwConsentData', array(
            'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
            'nonce'          => wp_create_nonce( 'cpw_consent_nonce' ),
            'cookieName'     => $cookie_name,
            'cookieDuration' => $cookie_duration,
        ) );
    }

    // -------------------------------------------------------------------------
    // Injeção do Google Consent Mode v2 no <head>
    // -------------------------------------------------------------------------

    public function inject_google_consent_mode() {
        $google_options = get_option( 'cpw_google_options', array() );

        if ( empty( $google_options['gcm_enable'] ) ) {
            return;
        }

        $gtm_id = ! empty( $google_options['gtm_id'] ) ? sanitize_text_field( $google_options['gtm_id'] ) : '';
        $ga4_id = ! empty( $google_options['ga4_id'] ) ? sanitize_text_field( $google_options['ga4_id'] ) : '';

        $analytics_default       = ! empty( $google_options['gcm_default_analytics'] ) ? $google_options['gcm_default_analytics'] : 'denied';
        $ad_storage_default      = ! empty( $google_options['gcm_default_ad_storage'] ) ? $google_options['gcm_default_ad_storage'] : 'denied';
        $ad_user_data_default    = ! empty( $google_options['gcm_default_ad_user_data'] ) ? $google_options['gcm_default_ad_user_data'] : 'denied';
        $ad_personal_default     = ! empty( $google_options['gcm_default_ad_personalization'] ) ? $google_options['gcm_default_ad_personalization'] : 'denied';
        $functionality_default   = ! empty( $google_options['gcm_default_functionality'] ) ? $google_options['gcm_default_functionality'] : 'granted';
        $security_default        = ! empty( $google_options['gcm_default_security'] ) ? $google_options['gcm_default_security'] : 'granted';

        ?>
<!-- CPW Consentimento - Google Consent Mode v2 -->
<script>
window.dataLayer = window.dataLayer || [];
function gtag(){dataLayer.push(arguments);}

// Define os estados padrão ANTES do carregamento das tags (obrigatório para LGPD)
gtag('consent', 'default', {
    'analytics_storage':      '<?php echo esc_js( $analytics_default ); ?>',
    'ad_storage':             '<?php echo esc_js( $ad_storage_default ); ?>',
    'ad_user_data':           '<?php echo esc_js( $ad_user_data_default ); ?>',
    'ad_personalization':     '<?php echo esc_js( $ad_personal_default ); ?>',
    'functionality_storage':  '<?php echo esc_js( $functionality_default ); ?>',
    'security_storage':       '<?php echo esc_js( $security_default ); ?>',
    'personalization_storage': 'denied',
    'wait_for_update': 500
});

// Restaura o consentimento salvo anteriormente
(function() {
    var cookieName = '<?php echo esc_js( ! empty( get_option( 'cpw_general_options', array() )['cookie_name'] ) ? get_option( 'cpw_general_options', array() )['cookie_name'] : 'cpw_consent_status' ); ?>';
    var consentDataCookie = 'cpw_consent_data';

    function getCookie(name) {
        var nameEQ = name + '=';
        var ca = document.cookie.split(';');
        for (var i = 0; i < ca.length; i++) {
            var c = ca[i];
            while (c.charAt(0) === ' ') c = c.substring(1, c.length);
            if (c.indexOf(nameEQ) === 0) return decodeURIComponent(c.substring(nameEQ.length, c.length));
        }
        return null;
    }

    var savedConsent = getCookie(cookieName);
    if (savedConsent !== null) {
        var savedData = getCookie(consentDataCookie);
        if (savedData) {
            try {
                var data = JSON.parse(savedData);
                gtag('consent', 'update', data);
            } catch(e) {}
        }
    }
})();
</script>
<?php

        // Injetar Google Tag Manager
        if ( $gtm_id ) {
            ?>
<!-- Google Tag Manager -->
<script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':
new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],
j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=
'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);
})(window,document,'script','dataLayer','<?php echo esc_js( $gtm_id ); ?>');</script>
<!-- End Google Tag Manager -->
            <?php
        }

        // Injetar Google Analytics 4 (se não usar GTM)
        if ( $ga4_id && ! $gtm_id ) {
            ?>
<!-- Google Analytics 4 -->
<script async src="https://www.googletagmanager.com/gtag/js?id=<?php echo esc_attr( $ga4_id ); ?>"></script>
<script>
window.dataLayer = window.dataLayer || [];
function gtag(){dataLayer.push(arguments);}
gtag('js', new Date());
gtag('config', '<?php echo esc_js( $ga4_id ); ?>');
</script>
<!-- End Google Analytics 4 -->
            <?php
        }
    }

    // -------------------------------------------------------------------------
    // Injeção do noscript do GTM no <body>
    // -------------------------------------------------------------------------

    public function inject_gtm_body() {
        $google_options = get_option( 'cpw_google_options', array() );
        if ( empty( $google_options['gcm_enable'] ) || empty( $google_options['gtm_id'] ) ) {
            return;
        }
        $gtm_id = sanitize_text_field( $google_options['gtm_id'] );
        ?>
<!-- Google Tag Manager (noscript) -->
<noscript><iframe src="https://www.googletagmanager.com/ns.html?id=<?php echo esc_attr( $gtm_id ); ?>"
height="0" width="0" style="display:none;visibility:hidden"></iframe></noscript>
<!-- End Google Tag Manager (noscript) -->
        <?php
    }

    // -------------------------------------------------------------------------
    // Exibição do banner
    // -------------------------------------------------------------------------

    public function add_consent_banner() {
        $general_options = get_option( 'cpw_general_options', array() );
        if ( empty( $general_options['enable_plugin'] ) ) {
            return;
        }
        require_once plugin_dir_path( __FILE__ ) . 'partials/banner.php';
    }

    // -------------------------------------------------------------------------
    // AJAX: Registrar consentimento
    // -------------------------------------------------------------------------

    public function ajax_log_consent() {
        check_ajax_referer( 'cpw_consent_nonce', 'nonce' );

        $privacy_options = get_option( 'cpw_privacy_options', array() );
        if ( empty( $privacy_options['enable_consent_log'] ) ) {
            wp_send_json_success( array( 'message' => 'Logging disabled' ) );
            return;
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'cpw_consent_logs';

        $consent_status = isset( $_POST['consent_status'] ) ? sanitize_text_field( $_POST['consent_status'] ) : 'unknown';
        $consent_data   = isset( $_POST['consent_data'] ) ? sanitize_text_field( $_POST['consent_data'] ) : '{}';

        $data_array = json_decode( $consent_data, true );
        if ( ! is_array( $data_array ) ) {
            $data_array = array();
        }

        // Anonimização do IP conforme LGPD
        $ip = $_SERVER['REMOTE_ADDR'] ?? '';
        $ip_parts = explode('.', $ip);
        if ( count( $ip_parts ) === 4 ) {
            $ip_parts[3] = '0';
            $ip = implode('.', $ip_parts);
        }

        $wpdb->insert(
            $table_name,
            array(
                'ip_address'           => $ip,
                'consent_status'       => $consent_status,
                'analytics_storage'    => isset( $data_array['analytics_storage'] ) ? $data_array['analytics_storage'] : 'denied',
                'ad_storage'           => isset( $data_array['ad_storage'] ) ? $data_array['ad_storage'] : 'denied',
                'functionality_storage' => isset( $data_array['functionality_storage'] ) ? $data_array['functionality_storage'] : 'granted',
                'user_agent'           => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( $_SERVER['HTTP_USER_AGENT'] ) : '',
                'created_at'           => current_time( 'mysql' ),
            ),
            array( '%s', '%s', '%s', '%s', '%s', '%s', '%s' )
        );

        wp_send_json_success( array( 'message' => 'Consent logged' ) );
    }
}
