<?php
/**
 * Classe responsável pelo painel de administração do plugin CPW Consentimento.
 *
 * @package CPW_Consentimento
 * @subpackage CPW_Consentimento/admin
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class CPW_Admin {

    private $plugin_name;
    private $version;

    public function __construct( $plugin_name, $version ) {
        $this->plugin_name = $plugin_name;
        $this->version     = $version;
    }

    // -------------------------------------------------------------------------
    // Enqueue de estilos e scripts
    // -------------------------------------------------------------------------

    public function enqueue_styles( $hook ) {
        if ( strpos( $hook, $this->plugin_name ) === false ) {
            return;
        }
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'css/admin-style.css',
            array(),
            $this->version,
            'all'
        );
    }

    public function enqueue_scripts( $hook ) {
        if ( strpos( $hook, $this->plugin_name ) === false ) {
            return;
        }
        wp_enqueue_script(
            $this->plugin_name,
            plugin_dir_url( __FILE__ ) . 'js/admin-script.js',
            array( 'jquery', 'wp-color-picker' ),
            $this->version,
            true
        );
        wp_enqueue_style( 'wp-color-picker' );
    }

    // -------------------------------------------------------------------------
    // Menu de administração
    // -------------------------------------------------------------------------

    public function add_plugin_admin_menu() {
        add_menu_page(
            __( 'CPW Consentimento', 'cpw-consentimento' ),
            __( 'CPW Consentimento', 'cpw-consentimento' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_general_settings' ),
            'dashicons-privacy',
            80
        );

        add_submenu_page(
            $this->plugin_name,
            __( 'Configurações Gerais', 'cpw-consentimento' ),
            __( 'Configurações Gerais', 'cpw-consentimento' ),
            'manage_options',
            $this->plugin_name,
            array( $this, 'display_general_settings' )
        );

        add_submenu_page(
            $this->plugin_name,
            __( 'Google Consent Mode', 'cpw-consentimento' ),
            __( 'Google Consent Mode', 'cpw-consentimento' ),
            'manage_options',
            $this->plugin_name . '-google',
            array( $this, 'display_google_settings' )
        );

        add_submenu_page(
            $this->plugin_name,
            __( 'Aparência do Banner', 'cpw-consentimento' ),
            __( 'Aparência do Banner', 'cpw-consentimento' ),
            'manage_options',
            $this->plugin_name . '-banner',
            array( $this, 'display_banner_settings' )
        );

        add_submenu_page(
            $this->plugin_name,
            __( 'Política de Privacidade', 'cpw-consentimento' ),
            __( 'Política de Privacidade', 'cpw-consentimento' ),
            'manage_options',
            $this->plugin_name . '-privacy',
            array( $this, 'display_privacy_settings' )
        );

        add_submenu_page(
            $this->plugin_name,
            __( 'Registros de Consentimento', 'cpw-consentimento' ),
            __( 'Registros de Consentimento', 'cpw-consentimento' ),
            'manage_options',
            $this->plugin_name . '-logs',
            array( $this, 'display_consent_logs' )
        );
    }

    // -------------------------------------------------------------------------
    // Registro das configurações via Settings API
    // -------------------------------------------------------------------------

    public function register_settings() {
        // --- Seção: Geral ---
        register_setting( 'cpw_general_options', 'cpw_general_options', array( $this, 'sanitize_general_options' ) );

        add_settings_section(
            'cpw_general_section',
            __( 'Configurações Gerais', 'cpw-consentimento' ),
            array( $this, 'general_section_callback' ),
            $this->plugin_name
        );

        add_settings_field(
            'cpw_enable_plugin',
            __( 'Ativar Plugin', 'cpw-consentimento' ),
            array( $this, 'render_checkbox_field' ),
            $this->plugin_name,
            'cpw_general_section',
            array( 'option_group' => 'cpw_general_options', 'field' => 'enable_plugin', 'label' => __( 'Habilitar o banner de consentimento no site', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_cookie_duration',
            __( 'Duração do Cookie (dias)', 'cpw-consentimento' ),
            array( $this, 'render_number_field' ),
            $this->plugin_name,
            'cpw_general_section',
            array( 'option_group' => 'cpw_general_options', 'field' => 'cookie_duration', 'default' => 365, 'min' => 1, 'max' => 3650 )
        );

        add_settings_field(
            'cpw_cookie_name',
            __( 'Nome do Cookie', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name,
            'cpw_general_section',
            array( 'option_group' => 'cpw_general_options', 'field' => 'cookie_name', 'default' => 'cpw_consent_status' )
        );

        add_settings_field(
            'cpw_block_scripts',
            __( 'Bloquear Scripts Antes do Consentimento', 'cpw-consentimento' ),
            array( $this, 'render_checkbox_field' ),
            $this->plugin_name,
            'cpw_general_section',
            array( 'option_group' => 'cpw_general_options', 'field' => 'block_scripts', 'label' => __( 'Bloquear scripts de rastreamento até o usuário consentir (recomendado pela LGPD)', 'cpw-consentimento' ) )
        );

        // --- Seção: Google Consent Mode ---
        register_setting( 'cpw_google_options', 'cpw_google_options', array( $this, 'sanitize_google_options' ) );

        add_settings_section(
            'cpw_google_section',
            __( 'Integração com Google Consent Mode v2', 'cpw-consentimento' ),
            array( $this, 'google_section_callback' ),
            $this->plugin_name . '-google'
        );

        add_settings_field(
            'cpw_gcm_enable',
            __( 'Ativar Google Consent Mode v2', 'cpw-consentimento' ),
            array( $this, 'render_checkbox_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array( 'option_group' => 'cpw_google_options', 'field' => 'gcm_enable', 'label' => __( 'Habilitar integração com o Google Consent Mode v2', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_gtm_id',
            __( 'ID do Google Tag Manager', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array( 'option_group' => 'cpw_google_options', 'field' => 'gtm_id', 'placeholder' => 'GTM-XXXXXXX', 'description' => __( 'Insira o ID do seu contêiner do Google Tag Manager (ex: GTM-XXXXXXX)', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_ga4_id',
            __( 'ID do Google Analytics 4 (GA4)', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array( 'option_group' => 'cpw_google_options', 'field' => 'ga4_id', 'placeholder' => 'G-XXXXXXXXXX', 'description' => __( 'Insira o ID de medição do GA4 (ex: G-XXXXXXXXXX)', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_gcm_default_analytics',
            __( 'Padrão: analytics_storage', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array(
                'option_group' => 'cpw_google_options',
                'field'        => 'gcm_default_analytics',
                'options'      => array( 'denied' => __( 'Negado (Recomendado - LGPD)', 'cpw-consentimento' ), 'granted' => __( 'Concedido', 'cpw-consentimento' ) ),
                'description'  => __( 'Estado padrão para armazenamento de dados de analytics antes do consentimento.', 'cpw-consentimento' )
            )
        );

        add_settings_field(
            'cpw_gcm_default_ad_storage',
            __( 'Padrão: ad_storage', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array(
                'option_group' => 'cpw_google_options',
                'field'        => 'gcm_default_ad_storage',
                'options'      => array( 'denied' => __( 'Negado (Recomendado - LGPD)', 'cpw-consentimento' ), 'granted' => __( 'Concedido', 'cpw-consentimento' ) ),
                'description'  => __( 'Estado padrão para armazenamento de cookies de publicidade.', 'cpw-consentimento' )
            )
        );

        add_settings_field(
            'cpw_gcm_default_ad_user_data',
            __( 'Padrão: ad_user_data', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array(
                'option_group' => 'cpw_google_options',
                'field'        => 'gcm_default_ad_user_data',
                'options'      => array( 'denied' => __( 'Negado (Recomendado - LGPD)', 'cpw-consentimento' ), 'granted' => __( 'Concedido', 'cpw-consentimento' ) ),
                'description'  => __( 'Estado padrão para envio de dados de usuário para publicidade.', 'cpw-consentimento' )
            )
        );

        add_settings_field(
            'cpw_gcm_default_ad_personalization',
            __( 'Padrão: ad_personalization', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array(
                'option_group' => 'cpw_google_options',
                'field'        => 'gcm_default_ad_personalization',
                'options'      => array( 'denied' => __( 'Negado (Recomendado - LGPD)', 'cpw-consentimento' ), 'granted' => __( 'Concedido', 'cpw-consentimento' ) ),
                'description'  => __( 'Estado padrão para personalização de anúncios.', 'cpw-consentimento' )
            )
        );

        add_settings_field(
            'cpw_gcm_default_functionality',
            __( 'Padrão: functionality_storage', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array(
                'option_group' => 'cpw_google_options',
                'field'        => 'gcm_default_functionality',
                'options'      => array( 'granted' => __( 'Concedido (Recomendado)', 'cpw-consentimento' ), 'denied' => __( 'Negado', 'cpw-consentimento' ) ),
                'description'  => __( 'Estado padrão para cookies de funcionalidade do site.', 'cpw-consentimento' )
            )
        );

        add_settings_field(
            'cpw_gcm_default_security',
            __( 'Padrão: security_storage', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-google',
            'cpw_google_section',
            array(
                'option_group' => 'cpw_google_options',
                'field'        => 'gcm_default_security',
                'options'      => array( 'granted' => __( 'Concedido (Recomendado)', 'cpw-consentimento' ), 'denied' => __( 'Negado', 'cpw-consentimento' ) ),
                'description'  => __( 'Estado padrão para cookies de segurança.', 'cpw-consentimento' )
            )
        );

        // --- Seção: Banner ---
        register_setting( 'cpw_banner_options', 'cpw_banner_options', array( $this, 'sanitize_banner_options' ) );

        add_settings_section(
            'cpw_banner_section',
            __( 'Personalização do Banner', 'cpw-consentimento' ),
            array( $this, 'banner_section_callback' ),
            $this->plugin_name . '-banner'
        );

        add_settings_field(
            'cpw_banner_title',
            __( 'Título do Banner', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'banner_title', 'default' => __( 'Sua privacidade é importante para nós', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_banner_text',
            __( 'Texto do Banner', 'cpw-consentimento' ),
            array( $this, 'render_textarea_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'banner_text', 'default' => __( 'Utilizamos cookies e tecnologias semelhantes para melhorar a sua experiência em nosso site, personalizar conteúdo e anúncios, e analisar o tráfego. Ao clicar em "Aceitar todos", você concorda com o uso de cookies. Você pode gerenciar suas preferências clicando em "Personalizar".', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_banner_position',
            __( 'Posição do Banner', 'cpw-consentimento' ),
            array( $this, 'render_select_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array(
                'option_group' => 'cpw_banner_options',
                'field'        => 'banner_position',
                'options'      => array(
                    'bottom'       => __( 'Rodapé (inferior)', 'cpw-consentimento' ),
                    'top'          => __( 'Cabeçalho (superior)', 'cpw-consentimento' ),
                    'bottom-left'  => __( 'Canto inferior esquerdo', 'cpw-consentimento' ),
                    'bottom-right' => __( 'Canto inferior direito', 'cpw-consentimento' ),
                    'center'       => __( 'Centro (modal)', 'cpw-consentimento' ),
                )
            )
        );

        add_settings_field(
            'cpw_banner_bg_color',
            __( 'Cor de Fundo', 'cpw-consentimento' ),
            array( $this, 'render_color_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'banner_bg_color', 'default' => '#ffffff' )
        );

        add_settings_field(
            'cpw_banner_text_color',
            __( 'Cor do Texto', 'cpw-consentimento' ),
            array( $this, 'render_color_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'banner_text_color', 'default' => '#333333' )
        );

        add_settings_field(
            'cpw_banner_accept_btn_color',
            __( 'Cor do Botão Aceitar', 'cpw-consentimento' ),
            array( $this, 'render_color_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'banner_accept_btn_color', 'default' => '#0073aa' )
        );

        add_settings_field(
            'cpw_banner_decline_btn_color',
            __( 'Cor do Botão Recusar', 'cpw-consentimento' ),
            array( $this, 'render_color_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'banner_decline_btn_color', 'default' => '#666666' )
        );

        add_settings_field(
            'cpw_accept_btn_text',
            __( 'Texto do Botão Aceitar', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'accept_btn_text', 'default' => __( 'Aceitar todos', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_decline_btn_text',
            __( 'Texto do Botão Recusar', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'decline_btn_text', 'default' => __( 'Recusar', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_customize_btn_text',
            __( 'Texto do Botão Personalizar', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-banner',
            'cpw_banner_section',
            array( 'option_group' => 'cpw_banner_options', 'field' => 'customize_btn_text', 'default' => __( 'Personalizar', 'cpw-consentimento' ) )
        );

        // --- Seção: Privacidade ---
        register_setting( 'cpw_privacy_options', 'cpw_privacy_options', array( $this, 'sanitize_privacy_options' ) );

        add_settings_section(
            'cpw_privacy_section',
            __( 'Política de Privacidade e LGPD', 'cpw-consentimento' ),
            array( $this, 'privacy_section_callback' ),
            $this->plugin_name . '-privacy'
        );

        add_settings_field(
            'cpw_privacy_page',
            __( 'Página de Política de Privacidade', 'cpw-consentimento' ),
            array( $this, 'render_page_select_field' ),
            $this->plugin_name . '-privacy',
            'cpw_privacy_section',
            array( 'option_group' => 'cpw_privacy_options', 'field' => 'privacy_page' )
        );

        add_settings_field(
            'cpw_dpo_name',
            __( 'Nome do Encarregado (DPO)', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-privacy',
            'cpw_privacy_section',
            array( 'option_group' => 'cpw_privacy_options', 'field' => 'dpo_name', 'description' => __( 'Nome do Encarregado de Proteção de Dados (Data Protection Officer) conforme exigido pela LGPD.', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_dpo_email',
            __( 'E-mail do Encarregado (DPO)', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-privacy',
            'cpw_privacy_section',
            array( 'option_group' => 'cpw_privacy_options', 'field' => 'dpo_email', 'description' => __( 'E-mail de contato do DPO para solicitações de titulares de dados.', 'cpw-consentimento' ) )
        );

        add_settings_field(
            'cpw_company_name',
            __( 'Nome da Empresa/Organização', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-privacy',
            'cpw_privacy_section',
            array( 'option_group' => 'cpw_privacy_options', 'field' => 'company_name' )
        );

        add_settings_field(
            'cpw_company_cnpj',
            __( 'CNPJ', 'cpw-consentimento' ),
            array( $this, 'render_text_field' ),
            $this->plugin_name . '-privacy',
            'cpw_privacy_section',
            array( 'option_group' => 'cpw_privacy_options', 'field' => 'company_cnpj', 'placeholder' => '00.000.000/0000-00' )
        );

        add_settings_field(
            'cpw_enable_consent_log',
            __( 'Registrar Consentimentos', 'cpw-consentimento' ),
            array( $this, 'render_checkbox_field' ),
            $this->plugin_name . '-privacy',
            'cpw_privacy_section',
            array( 'option_group' => 'cpw_privacy_options', 'field' => 'enable_consent_log', 'label' => __( 'Manter um registro de consentimentos para fins de auditoria (recomendado pela LGPD)', 'cpw-consentimento' ) )
        );
    }

    // -------------------------------------------------------------------------
    // Callbacks de seção
    // -------------------------------------------------------------------------

    public function general_section_callback() {
        echo '<p>' . esc_html__( 'Configure as opções gerais do plugin CPW Consentimento.', 'cpw-consentimento' ) . '</p>';
    }

    public function google_section_callback() {
        echo '<p>' . esc_html__( 'Configure a integração com o Google Consent Mode v2. Esta integração é necessária para que as tags do Google (Analytics, Ads) respeitem as escolhas de consentimento dos usuários, em conformidade com a LGPD.', 'cpw-consentimento' ) . '</p>';
        echo '<p><strong>' . esc_html__( 'Importante:', 'cpw-consentimento' ) . '</strong> ' . esc_html__( 'Todos os parâmetros são definidos como "Negado" por padrão, conforme exigido pela LGPD. As tags do Google utilizarão modelagem de conversão para preencher lacunas de dados.', 'cpw-consentimento' ) . '</p>';
    }

    public function banner_section_callback() {
        echo '<p>' . esc_html__( 'Personalize a aparência e o texto do banner de consentimento de cookies.', 'cpw-consentimento' ) . '</p>';
    }

    public function privacy_section_callback() {
        echo '<p>' . esc_html__( 'Configure as informações de privacidade e conformidade com a LGPD (Lei Geral de Proteção de Dados - Lei nº 13.709/2018).', 'cpw-consentimento' ) . '</p>';
    }

    // -------------------------------------------------------------------------
    // Renderização de campos
    // -------------------------------------------------------------------------

    public function render_text_field( $args ) {
        $options     = get_option( $args['option_group'], array() );
        $value       = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : ( isset( $args['default'] ) ? $args['default'] : '' );
        $placeholder = isset( $args['placeholder'] ) ? $args['placeholder'] : '';
        $description = isset( $args['description'] ) ? $args['description'] : '';
        printf(
            '<input type="text" name="%s[%s]" value="%s" placeholder="%s" class="regular-text" />',
            esc_attr( $args['option_group'] ),
            esc_attr( $args['field'] ),
            esc_attr( $value ),
            esc_attr( $placeholder )
        );
        if ( $description ) {
            echo '<p class="description">' . esc_html( $description ) . '</p>';
        }
    }

    public function render_textarea_field( $args ) {
        $options     = get_option( $args['option_group'], array() );
        $value       = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : ( isset( $args['default'] ) ? $args['default'] : '' );
        $description = isset( $args['description'] ) ? $args['description'] : '';
        printf(
            '<textarea name="%s[%s]" rows="5" class="large-text">%s</textarea>',
            esc_attr( $args['option_group'] ),
            esc_attr( $args['field'] ),
            esc_textarea( $value )
        );
        if ( $description ) {
            echo '<p class="description">' . esc_html( $description ) . '</p>';
        }
    }

    public function render_checkbox_field( $args ) {
        $options = get_option( $args['option_group'], array() );
        $checked = isset( $options[ $args['field'] ] ) ? checked( 1, $options[ $args['field'] ], false ) : '';
        $label   = isset( $args['label'] ) ? $args['label'] : '';
        printf(
            '<label><input type="checkbox" name="%s[%s]" value="1" %s /> %s</label>',
            esc_attr( $args['option_group'] ),
            esc_attr( $args['field'] ),
            $checked,
            esc_html( $label )
        );
    }

    public function render_select_field( $args ) {
        $options     = get_option( $args['option_group'], array() );
        $value       = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : ( isset( $args['default'] ) ? $args['default'] : '' );
        $description = isset( $args['description'] ) ? $args['description'] : '';
        printf( '<select name="%s[%s]">', esc_attr( $args['option_group'] ), esc_attr( $args['field'] ) );
        foreach ( $args['options'] as $key => $label ) {
            printf( '<option value="%s" %s>%s</option>', esc_attr( $key ), selected( $value, $key, false ), esc_html( $label ) );
        }
        echo '</select>';
        if ( $description ) {
            echo '<p class="description">' . esc_html( $description ) . '</p>';
        }
    }

    public function render_number_field( $args ) {
        $options = get_option( $args['option_group'], array() );
        $value   = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : ( isset( $args['default'] ) ? $args['default'] : '' );
        $min     = isset( $args['min'] ) ? $args['min'] : '';
        $max     = isset( $args['max'] ) ? $args['max'] : '';
        printf(
            '<input type="number" name="%s[%s]" value="%s" min="%s" max="%s" class="small-text" />',
            esc_attr( $args['option_group'] ),
            esc_attr( $args['field'] ),
            esc_attr( $value ),
            esc_attr( $min ),
            esc_attr( $max )
        );
    }

    public function render_color_field( $args ) {
        $options = get_option( $args['option_group'], array() );
        $value   = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : ( isset( $args['default'] ) ? $args['default'] : '#ffffff' );
        printf(
            '<input type="text" name="%s[%s]" value="%s" class="cpw-color-picker" data-default-color="%s" />',
            esc_attr( $args['option_group'] ),
            esc_attr( $args['field'] ),
            esc_attr( $value ),
            esc_attr( isset( $args['default'] ) ? $args['default'] : '#ffffff' )
        );
    }

    public function render_page_select_field( $args ) {
        $options = get_option( $args['option_group'], array() );
        $value   = isset( $options[ $args['field'] ] ) ? $options[ $args['field'] ] : '';
        $pages   = get_pages();
        printf( '<select name="%s[%s]">', esc_attr( $args['option_group'] ), esc_attr( $args['field'] ) );
        echo '<option value="">' . esc_html__( '-- Selecione uma página --', 'cpw-consentimento' ) . '</option>';
        foreach ( $pages as $page ) {
            printf( '<option value="%d" %s>%s</option>', $page->ID, selected( $value, $page->ID, false ), esc_html( $page->post_title ) );
        }
        echo '</select>';
        echo '<p class="description">' . esc_html__( 'Selecione a página que contém a sua Política de Privacidade.', 'cpw-consentimento' ) . '</p>';
    }

    // -------------------------------------------------------------------------
    // Sanitização
    // -------------------------------------------------------------------------

    public function sanitize_general_options( $input ) {
        $sanitized = array();
        $sanitized['enable_plugin']  = isset( $input['enable_plugin'] ) ? 1 : 0;
        $sanitized['cookie_duration'] = isset( $input['cookie_duration'] ) ? absint( $input['cookie_duration'] ) : 365;
        $sanitized['cookie_name']    = isset( $input['cookie_name'] ) ? sanitize_key( $input['cookie_name'] ) : 'cpw_consent_status';
        $sanitized['block_scripts']  = isset( $input['block_scripts'] ) ? 1 : 0;
        return $sanitized;
    }

    public function sanitize_google_options( $input ) {
        $sanitized = array();
        $sanitized['gcm_enable']                  = isset( $input['gcm_enable'] ) ? 1 : 0;
        $sanitized['gtm_id']                      = isset( $input['gtm_id'] ) ? sanitize_text_field( $input['gtm_id'] ) : '';
        $sanitized['ga4_id']                      = isset( $input['ga4_id'] ) ? sanitize_text_field( $input['ga4_id'] ) : '';
        $allowed_states                           = array( 'denied', 'granted' );
        $sanitized['gcm_default_analytics']       = isset( $input['gcm_default_analytics'] ) && in_array( $input['gcm_default_analytics'], $allowed_states ) ? $input['gcm_default_analytics'] : 'denied';
        $sanitized['gcm_default_ad_storage']      = isset( $input['gcm_default_ad_storage'] ) && in_array( $input['gcm_default_ad_storage'], $allowed_states ) ? $input['gcm_default_ad_storage'] : 'denied';
        $sanitized['gcm_default_ad_user_data']    = isset( $input['gcm_default_ad_user_data'] ) && in_array( $input['gcm_default_ad_user_data'], $allowed_states ) ? $input['gcm_default_ad_user_data'] : 'denied';
        $sanitized['gcm_default_ad_personalization'] = isset( $input['gcm_default_ad_personalization'] ) && in_array( $input['gcm_default_ad_personalization'], $allowed_states ) ? $input['gcm_default_ad_personalization'] : 'denied';
        $sanitized['gcm_default_functionality']   = isset( $input['gcm_default_functionality'] ) && in_array( $input['gcm_default_functionality'], $allowed_states ) ? $input['gcm_default_functionality'] : 'granted';
        $sanitized['gcm_default_security']        = isset( $input['gcm_default_security'] ) && in_array( $input['gcm_default_security'], $allowed_states ) ? $input['gcm_default_security'] : 'granted';
        return $sanitized;
    }

    public function sanitize_banner_options( $input ) {
        $sanitized = array();
        $sanitized['banner_title']          = isset( $input['banner_title'] ) ? sanitize_text_field( $input['banner_title'] ) : '';
        $sanitized['banner_text']           = isset( $input['banner_text'] ) ? wp_kses_post( $input['banner_text'] ) : '';
        $allowed_positions                  = array( 'bottom', 'top', 'bottom-left', 'bottom-right', 'center' );
        $sanitized['banner_position']       = isset( $input['banner_position'] ) && in_array( $input['banner_position'], $allowed_positions ) ? $input['banner_position'] : 'bottom';
        $sanitized['banner_bg_color']       = isset( $input['banner_bg_color'] ) ? sanitize_hex_color( $input['banner_bg_color'] ) : '#ffffff';
        $sanitized['banner_text_color']     = isset( $input['banner_text_color'] ) ? sanitize_hex_color( $input['banner_text_color'] ) : '#333333';
        $sanitized['banner_accept_btn_color'] = isset( $input['banner_accept_btn_color'] ) ? sanitize_hex_color( $input['banner_accept_btn_color'] ) : '#0073aa';
        $sanitized['banner_decline_btn_color'] = isset( $input['banner_decline_btn_color'] ) ? sanitize_hex_color( $input['banner_decline_btn_color'] ) : '#666666';
        $sanitized['accept_btn_text']       = isset( $input['accept_btn_text'] ) ? sanitize_text_field( $input['accept_btn_text'] ) : __( 'Aceitar todos', 'cpw-consentimento' );
        $sanitized['decline_btn_text']      = isset( $input['decline_btn_text'] ) ? sanitize_text_field( $input['decline_btn_text'] ) : __( 'Recusar', 'cpw-consentimento' );
        $sanitized['customize_btn_text']    = isset( $input['customize_btn_text'] ) ? sanitize_text_field( $input['customize_btn_text'] ) : __( 'Personalizar', 'cpw-consentimento' );
        return $sanitized;
    }

    public function sanitize_privacy_options( $input ) {
        $sanitized = array();
        $sanitized['privacy_page']        = isset( $input['privacy_page'] ) ? absint( $input['privacy_page'] ) : 0;
        $sanitized['dpo_name']            = isset( $input['dpo_name'] ) ? sanitize_text_field( $input['dpo_name'] ) : '';
        $sanitized['dpo_email']           = isset( $input['dpo_email'] ) ? sanitize_email( $input['dpo_email'] ) : '';
        $sanitized['company_name']        = isset( $input['company_name'] ) ? sanitize_text_field( $input['company_name'] ) : '';
        $sanitized['company_cnpj']        = isset( $input['company_cnpj'] ) ? sanitize_text_field( $input['company_cnpj'] ) : '';
        $sanitized['enable_consent_log']  = isset( $input['enable_consent_log'] ) ? 1 : 0;
        return $sanitized;
    }

    // -------------------------------------------------------------------------
    // Páginas de exibição
    // -------------------------------------------------------------------------

    public function display_general_settings() {
        require_once plugin_dir_path( __FILE__ ) . 'partials/settings-page.php';
    }

    public function display_google_settings() {
        require_once plugin_dir_path( __FILE__ ) . 'partials/google-settings-page.php';
    }

    public function display_banner_settings() {
        require_once plugin_dir_path( __FILE__ ) . 'partials/banner-settings-page.php';
    }

    public function display_privacy_settings() {
        require_once plugin_dir_path( __FILE__ ) . 'partials/privacy-settings-page.php';
    }

    public function display_consent_logs() {
        require_once plugin_dir_path( __FILE__ ) . 'partials/consent-logs-page.php';
    }
}
